// CrossMaker
// crossmaker.cpp

// includes
#include "c4d.h"
#include "crossmaker.h"
#include "Ocrossmaker.h"
#include "c4d_symbols.h"
#include "ge_math.h"

// CrossMaker class
Bool CrossMaker::Init(GeListNode *node)
{
	ResetValues(node);
	return true;
}

void CrossMaker::ResetValues(GeListNode *node)
{
	BaseObject *op = (BaseObject*)node;
	BaseContainer *data = op->GetDataInstance();

	data->SetFloat(CROSS_WIDTH, 200.0);
	data->SetFloat(CROSS_HEIGHT, 200.0);
	data->SetInt32(PRIM_PLANE, 0);
	data->SetBool(PRIM_REVERSE, false);
	data->SetInt32(SPLINEOBJECT_INTERPOLATION, SPLINEOBJECT_INTERPOLATION_ADAPTIVE);
	data->SetInt32(SPLINEOBJECT_SUB, 8);
	data->SetFloat(SPLINEOBJECT_ANGLE, Rad(5.0));
	data->SetFloat(SPLINEOBJECT_MAXIMUMLENGTH, 5.0);
}

Bool CrossMaker::Message(GeListNode *node, Int32 type, void *data)
{
	BaseDocument *doc;
	DescriptionCommand *desc;
	BaseContainer *nodebc;

	switch(type)
	{
	case MSG_MENUPREPARE:
		doc = (BaseDocument*)data;
		((BaseObject*)node)->GetDataInstance()->SetInt32(PRIM_PLANE, doc->GetSplinePlane());
		break;

	case MSG_DESCRIPTION_COMMAND:
		doc = node->GetDocument();
		desc = (DescriptionCommand*) data;
		nodebc = ((BaseObject*)node)->GetDataInstance();

		switch(desc->id[0].id)
		{
		case B_RESET_CROSS:
			doc->StartUndo();
			doc->AddUndo(UNDOTYPE_CHANGE, node);
			ResetValues(node);
			doc->EndUndo();
			node->Message(MSG_UPDATE);
			break;
		}
		break;
	}

	return true;
}

SplineObject* CrossMaker::GetContour(BaseObject *op, BaseDocument *doc, Float lod, BaseThread *bt)
{
	cCross cs;
	BaseContainer *bc = op->GetDataInstance();

	cs.crossHeight = bc->GetFloat(CROSS_HEIGHT);
	cs.crossWidth = bc->GetFloat(CROSS_WIDTH);
	SplineObject *so = GenerateCross(cs);
	if(!so) return nullptr;

	// get the base container of our new spline
	BaseContainer *bb = so->GetDataInstance();
	// now set its parameters using the parameters set for the plugin overall
	bb->SetInt32(SPLINEOBJECT_INTERPOLATION, bc->GetInt32(SPLINEOBJECT_INTERPOLATION));
	bb->SetInt32(SPLINEOBJECT_SUB, bc->GetInt32(SPLINEOBJECT_SUB));
	bb->SetFloat(SPLINEOBJECT_ANGLE, bc->GetFloat(SPLINEOBJECT_ANGLE));
	bb->SetFloat(SPLINEOBJECT_MAXIMUMLENGTH, bc->GetFloat(SPLINEOBJECT_MAXIMUMLENGTH));

	OrientObject(so, bc->GetInt32(PRIM_PLANE), bc->GetBool(PRIM_REVERSE));
	return so;
}

SplineObject* CrossMaker::GenerateCross(cCross ACross)
{
	Int32 pnts, i;
	Vector *padr;
	Float w, h;

	pnts = 12;
	SplineObject *op = SplineObject::Alloc(pnts, SPLINETYPE_LINEAR);
	if(!op)
		return nullptr;

	op->GetDataInstance()->SetBool(SPLINEOBJECT_CLOSED, true);
	padr = op->GetPointW();
	w = ACross.crossWidth;
	h = ACross.crossHeight;
	i = 0;
	padr[i++] = Vector((w/8) * -1, h/2, 0);		// point 0
	padr[i++] = Vector((w/8), h/2, 0);		// point 1
	padr[i++] = Vector((w/8), h/8, 0);		// point 2
	padr[i++] = Vector((w/2), h/8, 0);		// point 3
	padr[i++] = Vector((w/2), (h/8) * -1, 0);		// point 4
	padr[i++] = Vector((w/8), (h/8) * -1, 0);		// point 5
	padr[i++] = Vector((w/8), (h/2) * -1, 0);		// point 6
	padr[i++] = Vector((w/8) * -1, (h/2) * -1, 0);		// point 7
	padr[i++] = Vector((w/8) * -1, (h/8) * -1, 0);		// point 8
	padr[i++] = Vector((w/2) * -1, (h/8) * -1, 0);		// point 9
	padr[i++] = Vector((w/2) * -1, (h/8), 0);		// point 10
	padr[i++] = Vector((w/8) * -1, (h/8), 0);		// point 11


	op->Message(MSG_UPDATE);
	return op;
}



void CrossMaker::OrientObject(SplineObject *op, Int32 plane, Bool reverse)
{
	Vector *padr;
	Int32 pcnt, i;

	padr = op->GetPointW();
	pcnt = op->GetPointCount();
	if(!padr) return;

	if(plane >= 1)			// ZY or XZ planes
	{
		switch(plane)
		{
		case 1:				// ZY
			for(i = 0; i < pcnt; i++)
			{
				padr[i] = Vector(-padr[i].z, padr[i].y, padr[i].x);
			}
			break;

		case 2:				// XZ
			for(i = 0; i < pcnt; i++)
			{
				padr[i] = Vector(padr[i].x, -padr[i].z, padr[i].y);
			}
			break;
		}
	}

	if (reverse)
	{
		Vector	p;
		Int32 to = pcnt/2;
		if (pcnt%2) to++;
		for (i=0; i<to; i++)
		{
			p = padr[i]; padr[i] = padr[pcnt-1-i]; padr[pcnt-1-i] = p;
		}
	}

	op->Message(MSG_UPDATE);
}


// register the plugin
Bool RegisterCrossMaker(void)
{
	String plugname = GeLoadString(IDS_CROSSMAKER);
	return RegisterObjectPlugin(ID_CROSSMAKER, plugname, OBJECT_GENERATOR | OBJECT_ISSPLINE, CrossMaker::Alloc, "Ocrossmaker", AutoBitmap("crossmaker.tif"), 0);
}
