//////////////////////////////////////////////////////////////////////////
// Diamond ObjectData plugin
// diamond.cpp
//////////////////////////////////////////////////////////////////////////

#include "c4d.h"
#include "diamond.h"
#include "odiamond.h"
#include "c4d_symbols.h"

//////////////////////////////////////////////////////////////////////////
// Start of Diamond class implementation
//////////////////////////////////////////////////////////////////////////

Bool Diamond::Init(GeListNode *node)
{
	BaseObject *op = (BaseObject*)node;
	BaseContainer *data = op->GetDataInstance();

	// give the obejct a phong tag with appropriate values
	op->SetPhong(true, true, Rad(50.0));

	// initialise the description
	data->SetFloat(DIAMOND_WIDTH, DEFAULT_DIAMOND_WIDTH);
	data->SetFloat(DIAMOND_HEIGHT, DEFAULT_DIAMOND_HEIGHT);

	return true;
}

void Diamond::GetDimension(BaseObject *op, Vector *mp, Vector *rad)
{
	BaseContainer *bc = op->GetDataInstance();
	Float sizeH, sizeW;

	// we need to return the bounding box radius, so we divide the height and width by 2
	sizeH = bc->GetFloat(DIAMOND_HEIGHT) / 2;
	sizeW = bc->GetFloat(DIAMOND_WIDTH) / 2;

	// the centre of the box wil be the centre of the object
	*mp = Vector(0.0);
	// return the radius
	*rad = Vector(sizeW, sizeH, sizeW);
}

DRAWRESULT Diamond::Draw(BaseObject *op, DRAWPASS drawpass, BaseDraw *bd, BaseDrawHelp *bh)
{
	Int32 hitid, i;
	Matrix mx;
	HandleInfo info;

	if(drawpass != DRAWPASS_HANDLES)			// we only draw the grab handles in this plugin so for anything else, we just exit
		return DRAWRESULT_SKIP;

	hitid = op->GetHighlightHandle(bd);			// check if a handle is hit - that is, the user is hovering over it
	bd->SetMatrix_Matrix(op, bh->GetMg());		// converts coordinates into global space

	for(i = 0; i < HANDLES; i++)
	{
		if(i == hitid)							// if the handle is hit, draw it in a different color
			bd->SetPen(GetViewColor(VIEWCOLOR_SELECTION_PREVIEW));
		else
			bd->SetPen(GetViewColor(VIEWCOLOR_ACTIVEPOINT));
		GetHandle(op, i, info);					// get handle info for this handle so we know where to draw it
		bd->DrawHandle(info.position, DRAWHANDLE_BIG, 0);			// and draw the handle
	}
	// we've drawn the handles, now for the lines
	bd->SetPen(GetViewColor(VIEWCOLOR_ACTIVEPOINT));
	GetHandle(op, 0, info);
	bd->DrawLine(info.position, Vector(0.0), 0);	// draw the line from the centre of the object to the handle
	GetHandle(op, 1, info);
	bd->DrawLine(info.position, Vector(0.0), 0);	// and the other line

	return DRAWRESULT_OK;
}

Int32 Diamond::GetHandleCount(BaseObject *op)
{
	return HANDLES;									// for this object, there are only two handles
}

void Diamond::GetHandle(BaseObject *op, Int32 i, HandleInfo &info)
{
	BaseContainer *data = op->GetDataInstance();
	Float rectw, recth;

	// get the width and height and calculate the position of the handles
	rectw = data->GetFloat(DIAMOND_WIDTH);
	recth = data->GetFloat(DIAMOND_HEIGHT);

	switch(i)
	{
	case 0:
		info.position = Vector(rectw/2, 0, 0);		// this is the position of the width handle
		info.direction = Vector(1.0, 0.0, 0.0);
		info.type = HANDLECONSTRAINTTYPE_LINEAR;
		break;

	case 1:
		info.position = Vector(0.0, recth/2, 0.0);	// and of the height handle
		info.direction = Vector(0.0, 1.0, 0.0);
		info.type = HANDLECONSTRAINTTYPE_LINEAR;
		break;
	}
}

void Diamond::SetHandle(BaseObject *op, Int32 i, Vector p, const HandleInfo &info)
{
	BaseContainer *data = op->GetDataInstance();
	Float rad;

	// basically the reverse of GetHandle - set the width and height from a handle position
	rad = Dot(p, info.direction);
	switch(i)
	{
	case 0:
		data->SetFloat(DIAMOND_WIDTH, rad * 2);
		break;

	case 1:
		data->SetFloat(DIAMOND_HEIGHT, rad * 2);
		break;
	}
}

BaseObject* Diamond::GenerateObject(Vector *padr, Int32 pcnt, Int32 vcnt, BaseThread *bt)
{
	PolygonObject *op = nullptr;
	Vector *cpadr = nullptr;
	CPolygon *vadr = nullptr;
	UVWTag *uvtag = nullptr;
	UVWHandle uvwptr;
	UVWStruct us;
	Int32 i;

	op = PolygonObject::Alloc(pcnt, vcnt);		// allocate a new polygon object with the specified number of vertices and polygons
	if(op)
	{
		vadr = op->GetPolygonW();				// get the array of polygons
		cpadr = op->GetPointW();				// and the array of vertices
		// set point values
		for(i = 0; i < pcnt; i++)				// copy over the supplied array
		{
			cpadr[i] = padr[i];
		}
		// for each polygon, set the vertices which it uses (the vertex order is important to avoid reversed normals)
		vadr[0] = CPolygon(0, 1, 5, 5);
		vadr[1] = CPolygon(1, 2, 5, 5);
		vadr[2] = CPolygon(2, 3, 5, 5);
		vadr[3] = CPolygon(3, 0, 5, 5);
		vadr[4] = CPolygon(1, 0, 4, 4);
		vadr[5] = CPolygon(2, 1, 4, 4);
		vadr[6] = CPolygon(3, 2, 4, 4);
		vadr[7] = CPolygon(0, 3, 4, 4);

		// create a UV tag for use if the user makes the object editable
		uvtag = (UVWTag*)op->MakeVariableTag(Tuvw, op->GetPolygonCount());
		if(uvtag)
		{
			uvwptr = uvtag->GetDataAddressW();
			for(i = 0; i < op->GetPolygonCount(); i++)
			{
				us = UVWStruct(Vector(0.0, 0.0, 0.0), Vector(1.0, 0.0, 0.0), Vector(1.0, 1.0, 0.0), Vector(0.0, 1.0, 0.0));
				uvtag->Set(uvwptr, i, us);
			}
		}

		// return the generated object
		op->Message(MSG_UPDATE);
		return (BaseObject*)op;
	}
	else
		return nullptr;
}

BaseObject* Diamond::GetVirtualObjects(BaseObject *op, HierarchyHelp *hh)
{
	BaseContainer *bc = op->GetDataInstance();
	BaseObject *ret = nullptr;
	Vector *padr;
	Int32 numPoints, numPolys;
	Float xrad, yrad, zrad;

	// Cinema caches the object, so if we don't need to rebuild it, just return the cache
	Bool dirty = op->CheckCache(hh) || op->IsDirty(DIRTYFLAGS_DATA);
	if (!dirty) return op->GetCache(hh);

	// get the size - depth equals width in this case
	xrad = zrad = bc->GetFloat(DIAMOND_WIDTH)/2;
	yrad = bc->GetFloat(DIAMOND_HEIGHT)/2;

	numPoints = 6;		// our object has 6 vertices
	numPolys = 8;		// and 8 polygons (which are all triangles, but that doesn't matter)
	// allocate memory for the points array
	padr = NewMem(Vector, numPoints);
	if(!padr)
		return nullptr;	// out of memory
	// set the point values
	padr[0] = Vector(-xrad, 0.0, zrad);
	padr[1] = Vector(xrad, 0.0, zrad);
	padr[2] = Vector(xrad, 0.0, -zrad);
	padr[3] = Vector(-xrad, 0.0, -zrad);
	padr[4] = Vector(0.0, -yrad, 0.0);
	padr[5] = Vector(0.0, yrad, 0.0);

	// build our polygon object - doesn't really need a separate function but it makes the code easier to follow
	ret = GenerateObject(padr, numPoints, numPolys, hh->GetThread());
	DeleteMem(padr);		// don't need the allocated array any more

	if(ret)				// did we succeed in generating an object?
	{
		ret->KillTag(Tphong);			// probably not necessary since it won't have a phong tag, but...
		op->CopyTagsTo(ret, true, false, false, nullptr);		// copy over thetag we created in Init()
		ret->Message(MSG_UPDATE);		// tell Cinema the object has been changed
		return ret;						// return the new object for Cinema to show in the editor - we don't need to draw it
	}
	else				// something went wrong, so we allocate a new Null object (don't return nullptr - this is inefficient)
		return BaseObject::Alloc(Onull);
}

//////////////////////////////////////////////////////////////////////////
// End of Diamond class implementation
//////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////
// Other functions
//////////////////////////////////////////////////////////////////////////

Bool RegisterDiamond(void)
{
	// register our plugin with the system
	return RegisterObjectPlugin(ID_DIAMOND, GeLoadString(IDS_DIAMOND), OBJECT_GENERATOR, Diamond::Alloc, "odiamond", AutoBitmap("diamond.tif"), 0);
}
